/*
  isql -user SYSDBA -password masterkey "C:\Program Files (x86)\OOO REGOS SOFTWARE\REGOS - CASH SERVER\database\REGOS.fdb" -i aio_discount_n_plus_1.sql

  Discount N+1 trigger for STOCK_OPR_OPERATIONS_TMP.

  Logic:
	- For the affected document, reset discounted price (OPR_PRICE) to default price (OPR_PRICE2).
	- Compute QTY_SUM as SUM(OPR_QUANTITY) excluding rows where:
		OPR_HAS_STORNO is non-zero OR OPR_STORNO_UUID is not null/empty.
	- Also exclude rows where OPR_ITEM is in EXCLUDE_ITEM_IDS (comma-separated list).
	- DISCOUNT_PER_N = 4
	- DISCOUNT_FOR_N_ITEMS = floor(QTY_SUM / DISCOUNT_PER_N)
	- Apply 100% discount to the cheapest quantities across rows, handling OPR_QUANTITY > 1 by
	  adjusting per-unit OPR_PRICE so the line total reflects free units.
*/

SET TERM ^ ;

CREATE OR ALTER TRIGGER AIU_DISCOUNT_N_PLUS_1
FOR STOCK_OPR_OPERATIONS_TMP
ACTIVE AFTER INSERT OR UPDATE OR DELETE POSITION 0
AS
DECLARE VARIABLE V_DISCOUNT_PER_N       INTEGER        = 4;
/* Comma-separated list of item IDs to exclude from N+1 discount calculation (e.g. '12,34,56'). */
DECLARE VARIABLE EXCLUDE_ITEM_IDS       VARCHAR(4096)  = '';


DECLARE VARIABLE V_DOC                  TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_DOCUMENT;
DECLARE VARIABLE V_OPR_UUID             TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_UUID;
DECLARE VARIABLE V_QTY                  TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_QUANTITY;
DECLARE VARIABLE V_UNIT_PRICE           TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_PRICE;
DECLARE VARIABLE V_NEW_PRICE            TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_PRICE;

DECLARE VARIABLE V_QTY_SUM              NUMERIC(18, 4);
DECLARE VARIABLE V_DISCOUNT_FOR_N_ITEMS INTEGER;
DECLARE VARIABLE V_REMAINING_FREE_QTY   NUMERIC(18, 4);
DECLARE VARIABLE V_FREE_QTY             NUMERIC(18, 4);
BEGIN
	IF (RDB$GET_CONTEXT('USER_SESSION', 'DISCOUNT_GUARD') = '1') THEN
		EXIT;

	IF (INSERTING OR UPDATING) THEN
		V_DOC = NEW.OPR_DOCUMENT;
	ELSE
		V_DOC = OLD.OPR_DOCUMENT;

	IF (V_DOC IS NULL) THEN
		EXIT;

	RDB$SET_CONTEXT('USER_SESSION', 'DISCOUNT_GUARD', '1');

	/* Always reset prices first so discount is recalculated from scratch. */
	UPDATE STOCK_OPR_OPERATIONS_TMP S
	   SET S.OPR_PRICE = COALESCE(S.OPR_PRICE2, S.OPR_PRICE)
	 WHERE S.OPR_DOCUMENT = :V_DOC
	   AND (
			COALESCE(:EXCLUDE_ITEM_IDS, '') = ''
		OR POSITION(
			',' || CAST(S.OPR_ITEM AS VARCHAR(64)) || ','
			IN ',' || :EXCLUDE_ITEM_IDS || ','
		) = 0
	   )
	   AND NOT (
			COALESCE(S.OPR_HAS_STORNO, 0) <> 0
		OR COALESCE(S.OPR_STORNO_UUID, '') <> ''
	   );

	SELECT COALESCE(SUM(S.OPR_QUANTITY), 0)
	  FROM STOCK_OPR_OPERATIONS_TMP S
	 WHERE S.OPR_DOCUMENT = :V_DOC
	   AND S.OPR_QUANTITY > 0
	   AND (
			COALESCE(:EXCLUDE_ITEM_IDS, '') = ''
		OR POSITION(
			',' || CAST(S.OPR_ITEM AS VARCHAR(64)) || ','
			IN ',' || :EXCLUDE_ITEM_IDS || ','
		) = 0
	   )
	   AND NOT (
			COALESCE(S.OPR_HAS_STORNO, 0) <> 0
		OR COALESCE(S.OPR_STORNO_UUID, '') <> ''
	   )
	INTO :V_QTY_SUM;

	/* Take whole part (truncate), no rounding up/down. */
	V_DISCOUNT_FOR_N_ITEMS = CAST(TRUNC(:V_QTY_SUM / :V_DISCOUNT_PER_N) AS INTEGER);

	IF (COALESCE(V_DISCOUNT_FOR_N_ITEMS, 0) <= 0) THEN
	BEGIN
		RDB$SET_CONTEXT('USER_SESSION', 'DISCOUNT_GUARD', NULL);
		EXIT;
	END

	V_REMAINING_FREE_QTY = V_DISCOUNT_FOR_N_ITEMS;

	FOR
		SELECT
			S.OPR_UUID,
			S.OPR_QUANTITY,
			COALESCE(S.OPR_PRICE2, S.OPR_PRICE)
		  FROM STOCK_OPR_OPERATIONS_TMP S
		 WHERE S.OPR_DOCUMENT = :V_DOC
		   AND S.OPR_QUANTITY > 0
		   AND (
				COALESCE(:EXCLUDE_ITEM_IDS, '') = ''
			OR POSITION(
				',' || CAST(S.OPR_ITEM AS VARCHAR(64)) || ','
				IN ',' || :EXCLUDE_ITEM_IDS || ','
			) = 0
		   )
		   AND NOT (
				COALESCE(S.OPR_HAS_STORNO, 0) <> 0
			OR COALESCE(S.OPR_STORNO_UUID, '') <> ''
		   )
		 ORDER BY 3 ASC, 1 ASC
		INTO :V_OPR_UUID, :V_QTY, :V_UNIT_PRICE
	DO
	BEGIN
		IF (V_REMAINING_FREE_QTY <= 0) THEN
			LEAVE;

		/* Apply free quantity to this row (may be partial if V_QTY > remaining). */
		V_FREE_QTY = IIF(:V_QTY < :V_REMAINING_FREE_QTY, :V_QTY, :V_REMAINING_FREE_QTY);

		IF (V_FREE_QTY <= 0) THEN
			CONTINUE;

		/*
		  OPR_PRICE is treated as a per-unit price.
		  If we make K units free out of Q units, then new line total is (Q-K)*P.
		  To preserve totals using quantity*price, set new per-unit price to:
			  P * (Q-K) / Q
		*/
		V_NEW_PRICE = :V_UNIT_PRICE * (:V_QTY - :V_FREE_QTY) / :V_QTY;

		UPDATE STOCK_OPR_OPERATIONS_TMP S
		   SET S.OPR_PRICE = :V_NEW_PRICE
		 WHERE S.OPR_UUID = :V_OPR_UUID;

		V_REMAINING_FREE_QTY = :V_REMAINING_FREE_QTY - :V_FREE_QTY;
	END

	RDB$SET_CONTEXT('USER_SESSION', 'DISCOUNT_GUARD', NULL);
WHEN ANY DO
BEGIN
	RDB$SET_CONTEXT('USER_SESSION', 'DISCOUNT_GUARD', NULL);
	EXCEPTION;
END
END^

SET TERM ; ^
