/*
  isql -user SYSDBA -password masterkey "C:\Program Files (x86)\OOO REGOS SOFTWARE\REGOS - CASH SERVER\database\REGOS.fdb" -i aio_price_change_on_card.sql

  Price change trigger on card assignment for DOCS_DOC_SALES_REF.

  Logic:
    - When SLS_CARD changes and new value is not null:
        * set SLS_PRICE_TYPE = CARD_PRICE_TYPE_ID
        * update linked STOCK_OPR_OPERATIONS_TMP rows (OPR_DOCUMENT = SLS_UUID)
          to the price from CTLG_ITM_PRICES_REF for CARD_PRICE_TYPE_ID.
    - Guard rule: if SLS_CARD is null, SLS_PRICE_TYPE must be NO_CARD_PRICE_TYPE_ID.
*/

SET TERM ^ ;

/* Global config (edit these two values before running the script). */
EXECUTE BLOCK AS
BEGIN
	IF (NOT EXISTS(
		SELECT 1 FROM RDB$RELATIONS R
		 WHERE R.RDB$RELATION_NAME = 'AIO_PRICE_CHANGE_ON_CARD_CFG'
	)) THEN
	BEGIN
		EXECUTE STATEMENT
			'CREATE TABLE AIO_PRICE_CHANGE_ON_CARD_CFG (' ||
			'CFG_ID INTEGER NOT NULL, ' ||
			'NO_CARD_PRICE_TYPE_ID INTEGER, ' ||
			'CARD_PRICE_TYPE_ID INTEGER)';
		EXECUTE STATEMENT
			'ALTER TABLE AIO_PRICE_CHANGE_ON_CARD_CFG ' ||
			'ADD CONSTRAINT PK_AIO_PRICE_CHANGE_ON_CARD_CFG PRIMARY KEY (CFG_ID)';
	END
END^

EXECUTE BLOCK AS
DECLARE VARIABLE V_NO_CARD_PRICE_TYPE_ID INTEGER = 2; /* Вид цены без карты */
DECLARE VARIABLE V_CARD_PRICE_TYPE_ID    INTEGER = 1; /* Вид цены с картой */
BEGIN
	IF (EXISTS(
		SELECT 1 FROM AIO_PRICE_CHANGE_ON_CARD_CFG C WHERE C.CFG_ID = 1
	)) THEN
		UPDATE AIO_PRICE_CHANGE_ON_CARD_CFG C
		   SET C.NO_CARD_PRICE_TYPE_ID = :V_NO_CARD_PRICE_TYPE_ID,
			   C.CARD_PRICE_TYPE_ID = :V_CARD_PRICE_TYPE_ID
		 WHERE C.CFG_ID = 1;
	ELSE
		INSERT INTO AIO_PRICE_CHANGE_ON_CARD_CFG
			(CFG_ID, NO_CARD_PRICE_TYPE_ID, CARD_PRICE_TYPE_ID)
		VALUES
			(1, :V_NO_CARD_PRICE_TYPE_ID, :V_CARD_PRICE_TYPE_ID);
END^

/* Enforce SLS_CARD <-> SLS_PRICE_TYPE consistency before write. */
CREATE OR ALTER TRIGGER BIU_PRICE_TYPE_GUARD
FOR DOCS_DOC_SALES_REF
ACTIVE BEFORE INSERT OR UPDATE POSITION 0
AS
DECLARE VARIABLE V_NO_CARD_PRICE_TYPE_ID INTEGER;
DECLARE VARIABLE V_CARD_PRICE_TYPE_ID    INTEGER;
BEGIN
	SELECT C.NO_CARD_PRICE_TYPE_ID, C.CARD_PRICE_TYPE_ID
	  FROM AIO_PRICE_CHANGE_ON_CARD_CFG C
	 WHERE C.CFG_ID = 1
	INTO :V_NO_CARD_PRICE_TYPE_ID, :V_CARD_PRICE_TYPE_ID;

	/* If SLS_CARD is null, price type must be NO_CARD_PRICE_TYPE_ID. */
	IF (NEW.SLS_CARD IS NULL) THEN
		NEW.SLS_PRICE_TYPE = :V_NO_CARD_PRICE_TYPE_ID;
	ELSE
		NEW.SLS_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID;
END^

/* Update linked stock prices after card/price type is set. */
CREATE OR ALTER TRIGGER AIU_PRICE_CHANGE_ON_CARD
FOR DOCS_DOC_SALES_REF
ACTIVE AFTER INSERT OR UPDATE POSITION 0
AS
DECLARE VARIABLE V_NO_CARD_PRICE_TYPE_ID INTEGER;
DECLARE VARIABLE V_CARD_PRICE_TYPE_ID    INTEGER;

DECLARE VARIABLE V_DOC                 TYPE OF COLUMN DOCS_DOC_SALES_REF.SLS_UUID;
DECLARE VARIABLE V_NEW_CARD            TYPE OF COLUMN DOCS_DOC_SALES_REF.SLS_CARD;
BEGIN
	IF (RDB$GET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD') = '1') THEN
		EXIT;

	V_DOC = NEW.SLS_UUID;
	V_NEW_CARD = NEW.SLS_CARD;

	IF (V_DOC IS NULL) THEN
		EXIT;

	SELECT C.NO_CARD_PRICE_TYPE_ID, C.CARD_PRICE_TYPE_ID
	  FROM AIO_PRICE_CHANGE_ON_CARD_CFG C
	 WHERE C.CFG_ID = 1
	INTO :V_NO_CARD_PRICE_TYPE_ID, :V_CARD_PRICE_TYPE_ID;

	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', '1');

	/* If card is present, apply card price type to all linked rows. */
	IF (V_NEW_CARD IS NOT NULL) THEN
	BEGIN
		UPDATE STOCK_OPR_OPERATIONS_TMP S
		   SET S.OPR_PRICE  = (
				SELECT P.PRC_VALUE
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			),
			   S.OPR_PRICE2 = (
				SELECT P.PRC_VALUE
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			)
		 WHERE S.OPR_DOCUMENT = :V_DOC
		   AND EXISTS (
				SELECT 1
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			);
	END

	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', NULL);
WHEN ANY DO
BEGIN
	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', NULL);
	EXCEPTION;
END
END^

/* Apply card pricing to new/updated stock rows. */
CREATE OR ALTER TRIGGER AIU_PRICE_CHANGE_ON_OPR
FOR STOCK_OPR_OPERATIONS_TMP
ACTIVE AFTER INSERT OR UPDATE POSITION 0
AS
DECLARE VARIABLE V_CARD_PRICE_TYPE_ID    INTEGER;

DECLARE VARIABLE V_DOC                 TYPE OF COLUMN STOCK_OPR_OPERATIONS_TMP.OPR_DOCUMENT;
DECLARE VARIABLE V_HAS_CARD            INTEGER;
BEGIN
	IF (RDB$GET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD') = '1') THEN
		EXIT;

	V_DOC = NEW.OPR_DOCUMENT;

	IF (V_DOC IS NULL) THEN
		EXIT;

	SELECT IIF(D.SLS_CARD IS NULL, 0, 1)
	  FROM DOCS_DOC_SALES_REF D
	 WHERE D.SLS_UUID = :V_DOC
	INTO :V_HAS_CARD;

	IF (COALESCE(V_HAS_CARD, 0) = 0) THEN
		EXIT;

	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', '1');

	SELECT C.CARD_PRICE_TYPE_ID
	  FROM AIO_PRICE_CHANGE_ON_CARD_CFG C
	 WHERE C.CFG_ID = 1
	INTO :V_CARD_PRICE_TYPE_ID;

	UPDATE STOCK_OPR_OPERATIONS_TMP S
	   SET S.OPR_PRICE  = (
				SELECT P.PRC_VALUE
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			),
		   S.OPR_PRICE2 = (
				SELECT P.PRC_VALUE
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			)
	 WHERE S.OPR_UUID = NEW.OPR_UUID
	   AND EXISTS (
				SELECT 1
				  FROM CTLG_ITM_PRICES_REF P
				 WHERE P.PRC_ITEM = S.OPR_ITEM
				   AND P.PRC_PRICE_TYPE = :V_CARD_PRICE_TYPE_ID
			);

	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', NULL);
WHEN ANY DO
BEGIN
	RDB$SET_CONTEXT('USER_SESSION', 'PRICE_CHANGE_GUARD', NULL);
	EXCEPTION;
END
END^

SET TERM ; ^
